<?php

/**
 * The admin-specific functionality of the plugin.
 *
 * @link       https://thehp.in
 * @since      1.0.0
 *
 * @package    Global_Dns
 * @subpackage Global_Dns/admin
 */

/**
 * The admin-specific functionality of the plugin.
 *
 * Defines the plugin name, version, and two examples hooks for how to
 * enqueue the admin-specific stylesheet and JavaScript.
 *
 * @package    Global_Dns
 * @subpackage Global_Dns/admin
 */
class Global_Dns_Admin {

	/**
	 * The ID of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $plugin_name    The ID of this plugin.
	 */
	private $plugin_name;

	/**
	 * The version of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $version    The current version of this plugin.
	 */
	private $version;

	/**
	 * Initialize the class and set its properties.
	 *
	 * @since    1.0.0
	 * @param      string    $plugin_name       The name of this plugin.
	 * @param      string    $version    The version of this plugin.
	 */
	public function __construct($plugin_name, $version) {

		$this->plugin_name = $plugin_name;
		$this->version = $version;
	}

	/**
	 * Register the stylesheets for the admin area.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_styles() {

		/**
		 * This function is provided for demonstration purposes only.
		 *
		 * An instance of this class should be passed to the run() function
		 * defined in Global_Dns_Loader as all of the hooks are defined
		 * in that particular class.
		 *
		 * The Global_Dns_Loader will then create the relationship
		 * between the defined hooks and the functions defined in this
		 * class.
		 */
		if (strpos(get_current_screen()->id, $this->plugin_name) !== false) {
			wp_enqueue_style($this->plugin_name, plugin_dir_url(__FILE__) . 'css/global-dns-admin.min.css', array(), $this->version, 'all');
			wp_enqueue_style($this->plugin_name . 'overrider', plugin_dir_url(__FILE__) . 'css/default-override.min.css', array(), $this->version, 'all');
			wp_enqueue_style('animate.css', plugins_url($this->plugin_name) . '/vendor/animate/animate.min.css', array(), $this->version, 'all');
		}
	}

	/**
	 * Register the JavaScript for the admin area.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_scripts() {

		/**
		 * This function is provided for demonstration purposes only.
		 *
		 * An instance of this class should be passed to the run() function
		 * defined in Global_Dns_Loader as all of the hooks are defined
		 * in that particular class.
		 *
		 * The Global_Dns_Loader will then create the relationship
		 * between the defined hooks and the functions defined in this
		 * class.
		 */
		if (strpos(get_current_screen()->id, $this->plugin_name) !== false) {
			wp_enqueue_script($this->plugin_name, plugin_dir_url(__FILE__) . 'js/global-dns-admin.min.js', array(), $this->version, false);
			wp_enqueue_script('html5sortable', plugins_url($this->plugin_name) . '/vendor/html5sortable/html5sortable.min.js', array(), $this->version, false);
			wp_enqueue_script('toastify', plugins_url($this->plugin_name) . '/vendor/toastify/toastify.min.js', array(), $this->version, false);
		}
	}
	/**
	 * Admin Menu
	 * 
	 * @since	1.0.0
	 */
	public function add_plugin_admin_menu() {
		/*
		* Add a settings page for this plugin to the Settings menu.
		*
		* NOTE:  Alternative menu locations are available via WordPress administration menu functions.
		*
		*        Administration Menus: http://codex.wordpress.org/Administration_Menus
		*
		*/
		add_menu_page('Global DNS', 'Global DNS', 'global_dns', $this->plugin_name, array($this, 'display_global_dns_servers_page'), 'dashicons-admin-site');
		add_submenu_page($this->plugin_name, 'Servers', 'Servers', 'global_dns', $this->plugin_name, array($this, 'display_global_dns_servers_page'));
		add_submenu_page($this->plugin_name, 'Settings', 'Settings', 'global_dns', $this->plugin_name . '-settings', array($this, 'display_global_dns_settings_page'));
		add_submenu_page($this->plugin_name, 'Instructions', 'Instructions', 'global_dns', $this->plugin_name . '-instructions', array($this, 'display_global_dns_instructions_page'));
	}

	/**
	 * Clean the URL show on the servers page
	 * 
	 * @since	2.0.0
	 */
	public function cleanUrl($url) {
		$remove = ['https://', 'http://'];
		foreach ($remove as $item) {
			$url = str_replace($item, '', $url);
		}
		return $url;
	}

	/**
	 * Render the Servers Page
	 * 
	 * @since	1.0.0
	 */
	public function display_global_dns_servers_page() {
		include_once('partials/global-dns-admin-servers.php');
	}

	/**
	 * Validating Servers Data
	 * 
	 * @since	2.0.0
	 */
	public function validate_global_dns_servers() {
		$servers = get_option($this->plugin_name . "-servers");
		if (!is_array($servers)) {
			$servers = array();
		}
		$input = filter_input_array(INPUT_POST, FILTER_UNSAFE_RAW);
		if ($input['id'] != "" && ($input['id'] || $input['id'] == 0)) {
			$servers[$input['id']] = array(
				'name' => $input['name'],
				'url_ip' => $input['url_ip'],
				'latitude' => $input['latitude'],
				'longitude' => $input['longitude'],
				'country' => $input['country'],
				'meta' => $input['meta'],
			);
		} else {
			array_push($servers, [
				'name' => $input['name'],
				'url_ip' => $input['url_ip'],
				'latitude' => $input['latitude'],
				'longitude' => $input['longitude'],
				'country' => $input['country'],
				'meta' => $input['meta'],
			]);
		}
		return $servers;
	}

	/**
	 * Sort Servers
	 * 
	 * @since	2.2.0
	 */
	public function sort_global_dns_servers() {
		$servers = get_option($this->plugin_name . "-servers");
		if (!is_array($servers)) {
			$servers = array();
		}
		$input = filter_input_array(INPUT_POST, FILTER_UNSAFE_RAW);
		if (isset($input['from-id']) && isset($input['to-id'])) {
			$from = $input['from-id'];
			$to = $input['to-id'];
			$item = $servers[$from];
			unset($servers[$from]);
			$servers = array_values($servers);
			$servers = array_merge(array_slice($servers, 0, $to), array($item), array_slice($servers, $to));
		}
		return $servers;
	}

	/**
	 * Delete Server Data
	 * 
	 * @since	2.0.0
	 */
	public function delete_global_dns_server() {
		$servers = get_option($this->plugin_name . "-servers");
		if (!is_array($servers)) {
			$servers = array();
		}
		$input = filter_input_array(INPUT_POST, FILTER_UNSAFE_RAW);
		if (isset($input['delete-id'])) {
			if ($input['delete-id'] == 'all') {
				return [];
			}
			if (isset($servers[$input['delete-id']])) {
				unset($servers[$input['delete-id']]);
				$servers = array_values($servers);
			}
		}
		return $servers;
	}

	/**
	 * Handling Import of Servers
	 * 
	 * @since	2.0.0
	 */
	public function import_global_dns_servers() {
		$servers = get_option($this->plugin_name . "-servers");
		if (!is_array($servers)) {
			$servers = array();
		}
		$input = filter_input_array(INPUT_POST);
		if (isset($input['import-servers'])) {
			$imported = json_decode($input['import-servers'], true);
			if ($imported) {
				$servers = array_merge($servers, $imported);
			}
		}
		return $servers;
	}

	/**
	 * Saving Servers Data
	 * 
	 * @since	2.0.0
	 */
	public function save_global_dns_servers() {
		if (filter_input(INPUT_POST, 'servers', FILTER_UNSAFE_RAW) !== null) {
			register_setting($this->plugin_name, $this->plugin_name . '-servers', array($this, 'validate_global_dns_servers'));
		}
		if (filter_input(INPUT_POST, 'delete-id', FILTER_UNSAFE_RAW) !== null) {
			register_setting($this->plugin_name, $this->plugin_name . '-servers', array($this, 'delete_global_dns_server'));
		}
		if (filter_input(INPUT_POST, 'from-id', FILTER_UNSAFE_RAW) !== null) {
			register_setting($this->plugin_name, $this->plugin_name . '-servers', array($this, 'sort_global_dns_servers'));
		}
		if (filter_input(INPUT_POST, 'import-servers', FILTER_UNSAFE_RAW) !== null) {
			register_setting($this->plugin_name, $this->plugin_name . '-servers', array($this, 'import_global_dns_servers'));
		}
	}

	/**
	 * Render the Settings Page
	 * 
	 * @since	1.0.0
	 */
	public function display_global_dns_settings_page() {
		include_once('partials/global-dns-admin-settings.php');
	}

	/**
	 * Validating Settings Data
	 * 
	 * @since	2.0.0
	 */
	public function validate_global_dns_settings() {
		$input = filter_input_array(INPUT_POST, FILTER_UNSAFE_RAW);
		$data = array();
		$data['primary_color'] = $input['primary_color'];
		$data['enable_dark_mode'] = $input['enable_dark_mode'];
		$data['enable_expected_button'] = $input['enable_expected_button'];
		$data['dark_mode_color'] = $input['dark_mode_color'];
		$data['secret'] = $input['secret'];
		$data['timeout'] = $input['timeout'];
		$data['custom_css'] = $input['custom_css'];
		$data['blacklist_servers'] = $input['blacklist_servers'];
		return $data;
	}

	/**
	 * Saving Settings Data
	 * 
	 * @since	2.0.0
	 */
	public function save_global_dns_settings() {
		if (filter_input(INPUT_POST, 'settings', FILTER_UNSAFE_RAW) !== null) {
			register_setting($this->plugin_name, $this->plugin_name . '-settings', array($this, 'validate_global_dns_settings'));
		}
	}

	/**
	 * Render the Instructions Page
	 * 
	 * @since	1.0.0
	 */
	public function display_global_dns_instructions_page() {
		include_once('partials/global-dns-admin-instructions.php');
	}

	/**
	 * Show Success Message
	 * 
	 * @since	2.0.0
	 */
	public function show_admin_notice($type = 'success', $message = '') {
?>
		<div id="setting-error-settings_updated" class="notice notice-<?php esc_attr_e($type) ?> is-dismissible">
			<p><?php _e($message) ?></p>
		</div>
<?php
	}

	/**
	 * Get Admin Plugin directory
	 * 
	 * @since	2.0.0
	 */
	public function plugin_admin_dir() {
		return plugin_dir_url(__FILE__);
	}

	/**
	 * Check if current page is a AIO Contact Plugin Page
	 * 
	 * @since	2.0.0
	 */
	public function is_plugin_page() {
		$current_page_slug = isset($_GET['page']) ? sanitize_key($_GET['page']) : '';
		if (strpos($current_page_slug, 'global-dns') !== false) {
			return true;
		}
		return false;
	}

	/**
	 * Adding Plugin Header
	 * 
	 * @since	2.0.0
	 */
	public function add_plugin_header() {
		if ($this->is_plugin_page()) {
			include_once('partials/common/global-dns-admin-header.php');
		}
	}

	/**
	 * Changing the Footer Text for Plugin Pages
	 * 
	 * @since	2.0.0
	 */

	public function change_plugin_footer() {
		$url = "https://codecanyon.net/item/global-dns-multiple-server-dns-lookup/25869388";
		$text = '';
		if ($this->is_plugin_page()) {
			$text = sprintf(
				wp_kses(
					__('<span class="global-dns">Please rate <strong class="text-blue-500">Global DNS</strong> <a href="%1$s" class="text-blue-500" target="_blank" rel="noopener noreferrer">&#9733;&#9733;&#9733;&#9733;&#9733;</a> on <a href="%2$s" target="_blank" rel="noopener noreferrer">CodeCanyon.net</a> to give a super boost to our confidence in keep innovating. Thank you from <span class="font-bold text-green-600">thehp</span> Team!</span>', $this->plugin_name),
					array(
						'strong' => array(
							'class' => array(),
						),
						'span' => array(
							'class' => array(),
						),
						'a' => array(
							'class' => array(),
							'href'   => array(),
							'target' => array(),
							'rel'    => array(),
						),
					)
				),
				$url,
				$url
			);
		}
		return $text;
	}
}
